/*
 * @(#)OMOperation.java  1.0  6. Februar 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.oo.objectmodel;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import org.jhotdraw.xml.DOMStorable;

/**
 * Represents an operation defined by a {@link OMClass}.
 * <p>
 * Important: equals and hashCode must be based on identity and not on the
 * attributes of this object.
 * 
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.0 6. Februar 2004  Created.
 */
public class OMOperation implements Cloneable, DOMStorable {

    private String name;
    private OMClass clazz;
    private String method;
    private boolean isAbstract;

    /** Creates a new instance. */
    public OMOperation(OMClass clazz, String name) {
        this.clazz = clazz;
        this.name = name;
    }

    /**
     * This constructor is required for DOMStorable functionality.
     * Do <b>not</b> not use it for anything else.
     */
    public OMOperation() {
    }

    public OMClass getDeclaringClass() {
        return clazz;
    }

    public void setDeclaringClass(OMClass clazz) {
        this.clazz = clazz;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getName() {
        return name;
    }

    /**
     * Returns the operation using BeanShell syntax.
     */
    public String getOperationBsh() {
        StringBuilder buf = new StringBuilder();
        buf.append(getOperationName());
        buf.append('(');
        String[] paramNames = getParameterNames();
        for (int i = 0; i < paramNames.length; i++) {
            if (i != 0) {
                buf.append(',');
            }
            buf.append(paramNames[i]);
        }
        buf.append(')');
        return buf.toString();
    }

    /**
     * Returns an array with the parameter names.
     * Returns an empty array if the operation has no parameters.
     */
    public String[] getParameterNames() {
        int p = name.indexOf('(');
        if (p == -1 || p >= name.length() - 2) {
            return new String[0];
        }
        return name.substring(p + 1, name.length() - 1).split(",");
    }

    public String getOperationName() {
        int p = name.indexOf('(');
        return (p == -1) ? name : name.substring(0, p);
    }

    public Class[] getParameterTypes() {
        // XXX - Parse the parameter declaration of the operation and
        // fill the types into this array.
        return new Class[0];
    }

    public void setMethod(String method) {
        this.method = method;
    }

    public String getMethod() {
        return method;
    }

    public void setAbstract(boolean newValue) {
        if (isAbstract != newValue) {
            this.isAbstract = newValue;
//            fireAbstractChanged();
        }
    }

    public boolean isAbstract() {
        return isAbstract;
    }

    public Object clone() {
        try {
            OMOperation that = (OMOperation) super.clone();
            return that;
        } catch (CloneNotSupportedException e) {
            InternalError error = new InternalError(e.toString());
            //error.initCause(e); <- requires JDK 1.4
            throw error;
        }
    }

    public void read(DOMInput in) throws IOException {
        name = in.getAttribute("name", "operation()");
        isAbstract = in.getAttribute("abstract", false);

        if (in.getElementCount("method") > 0) {
            in.openElement("method");
            method = in.getText();
            in.closeElement();
        }
    }

    public void write(DOMOutput out) {
        out.addAttribute("name", name);
        out.addAttribute("abstract", isAbstract, false);
        if (method != null && method.length() > 0) {
            out.openElement("method");
            out.addText(method);
            out.closeElement();
        }
    }
}
